#include <stdio.h>
#include "thread.h"
#include "synch.h"

/**
 * This illustrates how we can avoid waking the
 * main thread more than once. Note, however, that
 * this requires duplicating the condition that
 * the main thread is waiting for. It also means
 * that it is not possible to use the condition
 * variable for more than one condition, and it
 * breaks the intuition that it is unproblematic
 * to wake "too many" threads by calling
 * cond_broadcast instead of cond_signal.
 *
 * The above drawbacks means that this approach is
 * typically not the first step in using condition
 * variables. However, in high-performance code,
 * this usage might be desirable to avoid waking
 * threads up needlessly.
 */

int result_a;
int result_b;
int threads_running;
struct lock threads_running_lock;
struct condition threads_running_cond;

void thread_fn_a(void) {
    result_a = 2;
    lock_acquire(&threads_running_lock);
    threads_running--;
    if (threads_running <= 0)
        cond_signal(&threads_running_cond,
                    &threads_running_lock);
    lock_release(&threads_running_lock);
}

void thread_fn_b(void) {
    result_b = 3;
    lock_acquire(&threads_running_lock);
    threads_running--;
    if (threads_running <= 0)
        cond_signal(&threads_running_cond,
                    &threads_running_lock);
    lock_release(&threads_running_lock);
}

int main(void) {
    lock_init(&threads_running_lock);
    cond_init(&threads_running_cond);
    threads_running = 2;

    thread_new(&thread_fn_a);
    thread_new(&thread_fn_b);

    lock_acquire(&threads_running_lock);
    if (threads_running > 0)
        cond_wait(&threads_running_cond,
                  &threads_running_lock);
    lock_release(&threads_running_lock);

    assert(result_a == 2);
    assert(result_b == 3);
    return 0;
}
