#include "os.h"
#include "thread.h"
#include "synch.h"
#include "timer.h"
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

// Does not need to do anything. Just being in a different translation unit is enough.
void prevent_optimization(void) {}

struct thread_args {
    void *run;
    void *aux[10];
};

struct thread_start;

typedef void (*trampoline_fn)(struct thread_start *);

struct thread_start {
    trampoline_fn trampoline;
    struct thread_args args;
    struct semaphore started;
};

static tid_t thread_start(struct thread_start *start);

static void thread_main_0(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void))a.run)();
}

static void thread_main_1(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void *))a.run)(a.aux[0]);
}

static void thread_main_2(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void *, void *))a.run)(a.aux[0], a.aux[1]);
}

static void thread_main_3(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void *, void *, void *))a.run)(a.aux[0], a.aux[1], a.aux[2]);
}

static void thread_main_4(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void *, void *, void *, void *))a.run)(a.aux[0], a.aux[1], a.aux[2], a.aux[3]);
}

static void thread_main_5(struct thread_start *start) {
    struct thread_args a = start->args;
    sema_up(&start->started);

    (*(void (*)(void *, void *, void *, void *, void *))a.run)(a.aux[0], a.aux[1], a.aux[2], a.aux[3], a.aux[4]);
}


tid_t thread_create(const char *name, int priority, thread_func *fn, void *aux) {
    struct thread_start start;
    start.trampoline = &thread_main_1;
    start.args.run = fn;
    start.args.aux[0] = aux;

    UNUSED(name);
    UNUSED(priority);

    return thread_start(&start);
}

tid_t thread_new_1(void *fn) {
    struct thread_start start;
    start.trampoline = &thread_main_0;
    start.args.run = fn;
    return thread_start(&start);
}

tid_t thread_new_2(void *fn, void *a) {
    struct thread_start start;
    start.trampoline = &thread_main_1;
    start.args.run = fn;
    start.args.aux[0] = a;
    return thread_start(&start);
}

tid_t thread_new_3(void *fn, void *a, void *b) {
    struct thread_start start;
    start.trampoline = &thread_main_2;
    start.args.run = fn;
    start.args.aux[0] = a;
    start.args.aux[1] = b;
    return thread_start(&start);
}

tid_t thread_new_4(void *fn, void *a, void *b, void *c) {
    struct thread_start start;
    start.trampoline = &thread_main_3;
    start.args.run = fn;
    start.args.aux[0] = a;
    start.args.aux[1] = b;
    start.args.aux[2] = c;
    return thread_start(&start);
}

tid_t thread_new_5(void *fn, void *a, void *b, void *c, void *d) {
    struct thread_start start;
    start.trampoline = &thread_main_4;
    start.args.run = fn;
    start.args.aux[0] = a;
    start.args.aux[1] = b;
    start.args.aux[2] = c;
    start.args.aux[3] = d;
    return thread_start(&start);
}

tid_t thread_new_6(void *fn, void *a, void *b, void *c, void *d, void *e) {
    struct thread_start start;
    start.trampoline = &thread_main_5;
    start.args.run = fn;
    start.args.aux[0] = a;
    start.args.aux[1] = b;
    start.args.aux[2] = c;
    start.args.aux[3] = d;
    start.args.aux[4] = e;
    return thread_start(&start);
}

#ifdef POSIX

static void *pthread_main(void *data) {
    struct thread_start *start = data;
    (*start->trampoline)(start);
    return NULL;
}

static tid_t thread_start(struct thread_start *start) {
    sema_init(&start->started, 0);

    pthread_t created;
    if (pthread_create(&created, NULL, &pthread_main, start)) {
        perror("pthread_create");
        exit(1);
    }

    sema_down(&start->started);
    sema_destroy(&start->started);

    // Get the same semantics as Pintos by calling detach.
    pthread_detach(created);

    return created;
}

tid_t thread_current(void) {
    return pthread_self();
}

void thread_exit(void) {
    pthread_exit(NULL);
}

void thread_yield(void) {
    sched_yield();
}

void timer_msleep(unsigned ms) {
    struct timespec time = { ms / 1000, (ms % 1000) * 1000000L };
    nanosleep(&time, NULL);
}

/**
 * Semaphore.
 */

void sema_init(struct semaphore *sema, unsigned value) {
    sem_init(&sema->os, 0, value);
}

void sema_destroy(struct semaphore *sema) {
    sem_destroy(&sema->os);
}

void sema_down(struct semaphore *sema) {
    while (sem_wait(&sema->os) != 0) {
        if (errno != EINTR) {
            perror("sem_wait");
            exit(1);
        }
    }
}

void sema_up(struct semaphore *sema) {
    sem_post(&sema->os);
}

/**
 * Lock.
 */

void lock_init(struct lock *lock) {
    pthread_mutexattr_t attr;
    pthread_mutexattr_init(&attr);
    pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK);
    pthread_mutex_init(&lock->os.mutex, &attr);
    pthread_mutexattr_destroy(&attr);
}

void lock_destroy(struct lock *lock) {
    pthread_mutex_destroy(&lock->os.mutex);
}

void lock_acquire(struct lock *lock) {
    int error = pthread_mutex_lock(&lock->os.mutex);
    switch (error) {
    case EINVAL:
        fprintf(stderr, "lock_acquire: the lock is not initialized properly\n");
        exit(1);
    case EDEADLK:
        fprintf(stderr, "lock_acquire: trying to acquire a lock that is already held\n");
        exit(1);
    }

    lock->os.owner = pthread_self();
}

void lock_release(struct lock *lock) {
    int error = pthread_mutex_unlock(&lock->os.mutex);
    switch (error) {
    case EPERM:
        fprintf(stderr, "lock_release: trying to release a lock that is not held by the current thread\n");
        exit(1);
    }
}

/**
 * Condition.
 */

void cond_init(struct condition *cond) {
    pthread_cond_init(&cond->os, NULL);
}

void cond_destroy(struct condition *cond) {
    pthread_cond_destroy(&cond->os);
}

static void check_lock_held(struct lock *lock, const char *fn) {
    int ok = pthread_mutex_trylock(&lock->os.mutex);
    switch (ok) {
    case 0:
        // Lock taken, release it.
        pthread_mutex_unlock(&lock->os.mutex);
        fprintf(stderr, "%s: the lock is not held by the current thread\n", fn);
        exit(1);
    case EBUSY:
        // Some thread holds the lock. It may or may not be us, depending on the
        // implementation.
    {
        pthread_t self = pthread_self();
        if (!pthread_equal(self, lock->os.owner)) {
            fprintf(stderr, "%s: the lock is not held by the current thread\n", fn);
            exit(1);
        }
        break;
    }
    case EINVAL:
        fprintf(stderr, "%s: the lock is not initialized properly\n", fn);
        exit(1);
    case EDEADLK:
        // This is actually the case that we want to happen. If the underlying
        // implementation gives this error, we are good to go without checking
        // the 'owner'.
        return;
    }
}

void cond_wait(struct condition *cond, struct lock *lock) {
    check_lock_held(lock, "cond_wait");

    int error = pthread_cond_wait(&cond->os, &lock->os.mutex);
    switch (error) {
    case EPERM: // This seems to be Linux-specific, not in the manpages. Seems to be exclusively for if the lock is not held.
    case EINVAL:
        fprintf(stderr, "cond_wait: the condition or lock is not initialized, or different locks are used with one condition\n");
        exit(1);
    }
}

void cond_signal(struct condition *cond, struct lock *lock) {
    check_lock_held(lock, "cond_signal");
    pthread_cond_signal(&cond->os);
}

void cond_broadcast(struct condition *cond, struct lock *lock) {
    check_lock_held(lock, "cond_broadcast");
    pthread_cond_broadcast(&cond->os);
}

/**
 * Timer.
 */

int64_t timer_ticks(void) {
    struct timespec time = { 0, 0 };
    clock_gettime(CLOCK_MONOTONIC, &time);

    return 1000000000*(int64_t)time.tv_sec + time.tv_nsec;
}

double timer_elapsed(int64_t since) {
    int64_t now = timer_ticks();

    return (now - since) / 1000000.0;
}

#endif


#ifdef WIN32

static DWORD WINAPI win32_start(void *data) {
    struct thread_start *start = data;
    (*start->trampoline)(start);
    return 0;
}

static tid_t thread_start(struct thread_start *start) {
    DWORD thread_id;
    HANDLE created;
    sema_init(&info.started, 0);

    created = CreateThread(NULL, 0, &win32_start, start, 0, &thread_id);
    if (!created) {
        printf("Failed creating thread %d\n", GetLastError());
        exit(1);
    }

    CloseHandle(created);

    sema_down(&start->started);
    sema_destroy(&start->started);

    return thread_id;
}

tid_t thread_current(void) {
    return GetCurrentThreadId();
}

void thread_exit(void) {
    ExitThread(0);
}

void thread_yield(void) {
    Sleep(0);
}

void timer_msleep(unsigned ms) {
    Sleep(ms);
}

/**
 * Semaphore.
 */

void sema_init(struct semaphore *sema, unsigned value) {
    // Note: We specify a value that should be large enough for 'maxCount'.
    sema->os = CreateSemaphore(NULL, value, 10000, NULL);
}

void sema_destroy(struct semaphore *sema) {
    CloseHandle(sema->os);
}

void sema_down(struct semaphore *sema) {
    WaitForSingleObject(sema->os, INFINITE);
}

void sema_up(struct semaphore *sema) {
    ReleaseSemaphore(sema->os, 1, NULL);
}

/**
 * Lock.
 */

void lock_init(struct lock *lock) {
    InitializeCriticalSection(&lock->os);
}

void lock_destroy(struct lock *lock) {
    DeleteCriticalSection(&lock->os);
}

void lock_acquire(struct lock *lock) {
    EnterCriticalSection(&lock->os);
}

void lock_release(struct lock *lock) {
    LeaveCriticalSection(&lock->os);
}

/**
 * Condition.
 */

void cond_init(struct condition *cond) {
    InitializeConditionVariable(&cond->os);
}

void cond_destroy(struct condition *cond) {
    // Nothing to do...
    UNUSED(cond);
}

void cond_wait(struct condition *cond, struct lock *lock) {
    SleepConditionVariableCS(&cond->os, &lock->os, INFINITE);
}

void cond_signal(struct condition *cond, struct lock *lock) {
    UNUSED(lock);
    WakeConditionVariable(&cond->os);
}

void cond_broadcast(struct condition *cond, struct lock *lock) {
    UNUSED(lock);
    WakeAllConditionVariable(&cond->os);
}

/**
 * Timer.
 */

int64_t timer_ticks(void) {
    LARGE_INTEGER out;
    QueryPerformanceCounter(&out);
    return out.QuadPart;
}

double timer_elapsed(int64_t since) {
    int64_t now = timer_ticks();
    LARGE_INTEGER freq;
    QueryPerformanceFrequency(&freq);

    return (now - since) / (freq.QuadPart / 1000.0);
}

#endif
